import java.util.*;

/*
 * Problem Statement:
 *
 * The N-Queens problem involves placing N queens on an N×N chessboard such that
 * no two queens threaten each other (i.e., No two queens can share the same
 * row, column, or diagonal).
 *
 */
public class NQueensState implements State {
    private int size;        // size of the board
    private int[] columns;   // indicates that queen at row r is placed at column columns[r]
    private int placed;      // number of queens that are already placed

    /*
     *  NQueensState constructor.
     */
    public NQueensState(int size, int[] columns, int placed) {
        this.size = size;
        this.columns = columns;
        this.placed = placed;
    }

    /*
     *  NQueensState initial state constructor.
     */
    public NQueensState(int size) {
        this.size = size;
        columns = new int[size];
        for (int i = 0; i < size; i ++) columns[i] = -1;
        placed = 0;
    }

    /*
     * A state is final if all queens are placed and no queen attacks another.
     */
    public boolean isGoal() {
        // all queens must be placed
        if (placed != size) return false;

        for (int r = 0; r < size; r++) {
            for (int i = 0; i < r; i++) {
                // are there two queens on the same column?
                if (columns[i] == columns[r]) return false;
                // are there two queens on the same diagonal?
                if (columns[r] - r == columns[i] - i) return false;
                if (columns[r] + r == columns[i] + i) return false;
            }
        }
        return true;
    }

    /*
     * Return all possible next states.
     */
    public Collection<State> next() {
        Collection<State> states = new ArrayList<State>();

        if (placed == size) return states;

        // iterate over placed queens to mark invalid placements
        boolean[] threat = new boolean[size];

        for (int r = 0; r < placed; r++) {
            // a queen on the same column is threatened
            threat[columns[r]] = true;

            // a queen on the same diagonal is threatened
            int diag1 = columns[r] - r + placed;
            int diag2 = columns[r] + r - placed;

            if (0 < diag1 && diag1 < size)
                threat[diag1] = true;
            if (0 < diag2 && diag2 < size)
                threat[diag2] = true;
        }

        // for each valid placement generate a new possible state
        for (int c = 0; c < size; c++) {
            if (!threat[c]) {
                int[] newcolums = Arrays.copyOf(columns, size);
                newcolums[placed] = c;

                NQueensState st = new NQueensState(size, newcolums, placed + 1);

                states.add(st);
            }
        }

        return states;
    }

    /*
     * Return parent state
     */
    public State getPrevious() {
        // we can keep track of the previous state, but it is not really needed
        return null;
    }


    /*
     * Override toSting to print a more informative representation.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (int r = 0; r < size; r++) {
            for (int c = 0; c < size; c++) {
                if (columns[r] == c) sb.append("X").append(" ");
                else sb.append("0").append(" ");
            }
            sb.append("\n");
        }
        return sb.toString();
    }


    /*
     * Override equals so that two states are equal if and only iff the four
     * entities have the same positions.
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof NQueensState)) return false;
        NQueensState other = (NQueensState) o;
        return Arrays.equals(this.columns, other.columns);
    }

    /*
     * Override hashCode so that if s1.equals(s2) then s1 and s2 have the same
     * hashes.
     */
    @Override
    public int hashCode() {
        return Arrays.hashCode(columns);
    }
}