import java.util.*;
import java.io.*;

/*
 * Problem Statement: The N-Puzzle is a sliding puzzle consisting of a grid with
 * N tiles numbered from 1 to N and one empty space. The goal is to rearrange
 * the tiles to reach a specific target configuration, typically with tiles
 * ordered in ascending order and the empty space at the bottom-right corner.
 * Tiles are rearranged by moving a tile to an adjacent empty space (up, down,
 * left, or right).
 *
 * Goal: Find the shortest sequence of moves that transforms the initial
 * configuration into the goal configuration. 
 *
 * For a more informed search algorithms that BFS, you can take a look at
 * https://www.cs.princeton.edu/courses/archive/fall17/cos226/assignments/8puzzle/index.html
 *
 */
public class NPuzzleState implements State {
    private int[][] board;
    private int size;
    private int emptyX;
    private int emptyY;
    private NPuzzleState prev;

    /*
     *  NPuzzleState constructor
     */
    public NPuzzleState(int[][] board, int emptyX, int emptyY, NPuzzleState prev) {
        this.board = board;
        this.size = board.length;
        this.emptyX = emptyX;
        this.emptyY = emptyY;
        this.prev = prev;
    }
    
    /*
     *  NPuzzleState initial state constructor
     */
    public NPuzzleState(int[][] board) {
        this.board = board;
        this.size = board.length;
        this.prev = null;
        // find the coordinates if the empty tile
        for (int i = 0; i < size; i++) {
            for (int j = 0; j < size; j++) {
                if (board[i][j] == 0) {
                    emptyX = i;
                    emptyY = j;
                    break;
                }
            }
        }
    }

    /*
     * A state is final if all tiles are in order and the last tile is the empty
     * one.
     */
    public boolean isGoal() {
        for (int i = 0; i < size; i++) {
            for (int j = 0; j < size; j++) {
                if (i == size - 1 && j == size - 1) {
                    if (board[i][j] != 0) return false;
                }
                else {
                    if (board[i][j] != j + i * size + 1) return false;
                }
            }
        }
        return true;
    }

    /*
     * Return all possible next states.
     */
    public Collection<State> next() {
        Collection<State> states = new ArrayList<State>();

        // Left
        if (emptyX > 0) {
            states.add(createNewState(emptyX - 1, emptyY));            
        }
        // Right
        if (emptyX < size - 1) {
            states.add(createNewState(emptyX + 1, emptyY));            
        }
        // Up
        if (emptyY > 0) {
            states.add(createNewState(emptyX, emptyY - 1));            
        }
        // Down
        if (emptyY < size - 1) {
            states.add(createNewState(emptyX, emptyY + 1));            
        }

        return states;
    }

    /*
     * Helper for creating a child state where empty is swapped with
     * the tile at position (x,y).
     */
    private NPuzzleState createNewState(int x, int y) {
        int[][] newBoard = deepCopy(board);
        
        newBoard[emptyX][emptyY] = newBoard[x][y];
        newBoard[x][y] = 0;

        return new NPuzzleState(newBoard, x, y, this);
    }


    /**
     * Utility to create a deep copy of a 2D array.
     */
    private int[][] deepCopy(int[][] original) {
        int[][] copy = new int[original.length][original[0].length];

        for (int i = 0; i < original.length; i++) {
            copy[i] = Arrays.copyOf(original[i], original[i].length);
        }
        return copy;
    }

    /*
     * Return the parent state.
     */
    public State getPrevious() {
        return prev;
    }


    /*
     * Override toSting to print a more informative representation.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (int r = 0; r < size; r++) {
            for (int c = 0; c < size; c++) {
                sb.append(board[r][c]).append(" ");
            }
            sb.append("\n");
        }
        return sb.toString();
    }
    

    /*
     * Override equals so that two states are equal if and only iff the four
     * entities have the same positions.
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof NPuzzleState)) return false;
        NPuzzleState other = (NPuzzleState) o;
        // Check all cells
        return Arrays.deepEquals(this.board, other.board);
        // for (int r = 0; r < size; r++) {
        //     for (int c = 0; c < size; c++) {
        //         if (this.board[r][c] != other.board[r][c]) return false;
        //     }
        // }
        // return true;
    }

    /*
     * Override hashCode so that if s1.equals(s2) then s1 and s2 have the same
     * hashes.
     */
    @Override
    public int hashCode() {
        return Arrays.deepHashCode(board);
    }


    /*
     * Class method to read a problem instance from a file given as argument.
     */
    public static NPuzzleState readInput(String file) {
        
        // input reading
        int size = 0;
        int[][] board = null;

        try {
            BufferedReader reader =
                new BufferedReader(new InputStreamReader(new FileInputStream(file)));
            Scanner sc = null;
            String line = null;

            line = reader.readLine();
            size = Integer.parseInt(line);
            board = new int[size][size];

            for (int i = 0; i < size; i++) {
                line = reader.readLine();
                sc = new Scanner(line);
                for (int j = 0; j < size; j++) board[i][j] = sc.nextInt();
            }

        } catch (Exception e) { // Bad practice!
            System.out.println("Something went wrong: " + e);
        }

        return new NPuzzleState(board);
    }

}
