import java.util.*;

/* 
 * Problem Statement: A man, a wolf, a goat, and a cabbage are on the east side
 * of a river. There is a boat at the riverbank that can carry the man and at
 * most one other entity at a time. Only the man can operate the boat.
 *
 * Constraints:
 * - The wolf cannot be left alone with the goat without the man present.
 * - The goat cannot be left alone with the cabbage without the man present.
 *
 * Goal: Find the sequence of moves that transfers all four entities to the west
 * side of the river using the minimum number of moves, without violating any of
 * the constraints.
 */
public class GoatState implements State {
    // represent all four entities as booleans where false = west, true = east
    private boolean man, wolf, goat, cabbage;
    // link to the parent state
    private State previous;

    /*
     * GoatState constructor
     */
    public GoatState(boolean m, boolean w, boolean g, boolean c, State prev) {
        man = m; wolf = w; goat = g; cabbage = c;
        previous = prev;
    }

    /*
     * GoatState constructor for the initial state only, using constructor overloading.
     */
    public GoatState() {
        man = false; wolf = false; goat = false; cabbage = false;
        previous = null;
    }

    /*
     * A state is final if all four entities are on the east side.
     */
    @Override
    public boolean isGoal() {
        return man && wolf && goat && cabbage;
    }

    /*
     * A state is valid if the wold and the goat are not on the same side
     * unattended (i.e. the man is not on the same side as them) and if the goat
     * and the cabbage are not on the same side unattended.
     */
    private boolean isValid() {
        return !(wolf == goat && man != goat)
            && !(goat == cabbage && man != cabbage);
    }

    /*
     * Return all possible valid next states.
     */
    @Override
    public Collection<State> next() {
        Collection<State> states = new ArrayList<State>();
        GoatState state = null;
        
        state = new GoatState(!man, wolf, goat, cabbage, this);
        if (state.isValid()) states.add(state);

        if (man == goat) {
            state = new GoatState(!man, wolf, !goat, cabbage, this);
            if (state.isValid()) states.add(state);
        }
        if (man == wolf) {
            state = new GoatState(!man, !wolf, goat, cabbage, this);
            if (state.isValid()) states.add(state);
        }
        if (man == cabbage) {
            state = new GoatState(!man, wolf, goat, !cabbage, this);
            if (state.isValid()) states.add(state);
        }

        return states;
    }

    /*
     * Return the parent state.
     */
    @Override
    public State getPrevious() {
        return previous;
    }

    /*
     * Override toSting to print a more informative representation.
     */
    @Override
    public String toString() {
       StringBuilder sb = new StringBuilder("State: ");
       sb.append("man=").append(man ? "E" : "W");
       sb.append(", wolf=").append(wolf ? "E" : "W");
       sb.append(", goat=").append(goat ? "E" : "W");
       sb.append(", cabbage=").append(cabbage ? "E" : "W");

       return sb.toString();
    }

    /*
     * Override equals so that two states are equal if and only iff the four
     * entities have the same positions.
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        GoatState s = (GoatState) o;
        return s.man == man && s.goat == goat && s.cabbage == cabbage && s.wolf == wolf;
    
    }

    /*
     * Override hashCode so that if s1.equals(s2) then s1 and s2 have the same
     * hashes.
     */
    @Override
    public int hashCode() {
        return Objects.hash(man, wolf, goat, cabbage);
    }

}